;*****************************************************************************
;  This code was designed to demonstrate how the 2444 could be interfaced to *
;  the 8031 microcontroller.  The interface uses 3 lines from Port 1 (P1.7,  *
;  P1.6, and P1.5) to communicate.  The DI and DO pins on the 2444 are tied  *
;  together which allows 1 less port line to be used.                        *
;                                                                            *
;  The code shown demonstrates RCL, WREN, READ, WRITE, and STORE             *
;  instructions.  The remaining instructions (WRDS and SLEEP) can be issued  *
;  using the same routine as other non-data instructions.                    *
;                                                                            *
;  The program issues a sequence of instructions to read the contents of     *
;  address 5 and stores the same value in address 9.  The sequence of        *
;  instructions is as follows :                                              *
;    1. RCL        sets the previous recall latch                            *
;    2. WREN       sets the write enable latch                               *
;    3. READ       data from address 5 is read                               *
;    4. WRITE      the data read during step 3 is written to address 9       *
;    5. STO        the ram's contents is transfered to the EEPROM            *
;                                                                            *
;  Data transfer is performed with the most significant bit first.  During   *
;  the READ and WRITE instructions the data sequence is inverted from that   *
;  shown in the data book (D15 is shifted first).                            *
;*****************************************************************************

SK	BIT	P1.5		; SK GENERATED ON BIT 5 OF PORT 1
CE	BIT     P1.6		; CE GENERATED ON BIT 6 OF PORT 1
DIO	BIT     P1.7		; DI AND DO GENERATED ON BIT 7 OF PORT 1

WRDS	EQU	80H 		; RESET WRITE ENABLE LATCH
STO	EQU	81H		; TRANSFERS FROM RAM TO EEPROM
SLEEP	EQU	82H		; PLACES PART INTO POWER DOWN MODE
WRITE	EQU	83H		; RAM WRITE
WREN	EQU	84H		; SET WRITE ENABLE LATCH
RCL	EQU	85H		; TRANSFERS FROM EEPROM TO RAM, RESETS
				;   WRITE ENABLE LATCH
READ	EQU	86H		; RAM READ

DATAHI	EQU	R0		; REGISTER WHERE HIGH ORDER DATA IS STORED
DATALO	EQU	R1		; REGISTER WHERE LOW ORDER DATA IS STORED
ADDR	EQU	R2		; REGISTER WHERE THE PART ADDRESS IS STORED
COUNT	EQU	R3		; LOOP COUNT VARIABLE
INST	EQU	R4		; REGISTER WHERE THE INSTRUCTION IS STORED

;********************************************
; RESET VECTOR TO BEGINNING OF PROGRAM CODE *
;********************************************

	ORG	0000H           ; RESET VECTORS TO THIS LOCATION
	LJMP	BEGIN

;*****************************
; START OF PROGRAM EXECUTION *
;*****************************

	ORG	0100H		; BEGINNING OF PROGRAM CODE
BEGIN:  MOV	SP,#60H		; INITIALIZE STACK POINTER

	CLR	CE		; INITIALIZE CONTROL SIGNALS TO 0
	CLR	SK
	CLR	DIO

	MOV	A,#RCL		; PERFORM A RECALL TO SET
	MOV	INST,A		;  THE RECALL LATCH
	LCALL	CEHIGH
	LCALL   OUTBYT
	LCALL	CELOW

	MOV	A,#WREN		; PERFORM A WRITE ENABLE TO SET
	MOV	INST,A		;  THE WRITE ENABLE LATCH
	LCALL	CEHIGH
	LCALL   OUTBYT
	LCALL	CELOW

	MOV	A,#05H		; READ THE CONTENTS OF ADDRESS 5
	MOV	ADDR,A		;  THE VALUE READ WILL BE IN STORED
	LCALL	RDWRD		;  IN DATALO AND DATAHI

	MOV	A,#09H          ; WRITE THE DATA JUST READ INTO
	MOV	ADDR,A	 	; ADDRESS 9
	LCALL	WRWRD

	MOV	A,#STO		; PERFORM A STORE OPERATION
	MOV	INST,A
	LCALL	CEHIGH
	LCALL	OUTBYT
	LCALL	CELOW

DONE:	LJMP	DONE            ; LOOP UNTIL RESET

;**************************************************************
; WRITE THE WORD SPECIFIED IN DATAHI AND DATALO.  THE ADDRESS *
;  TO BE WRITTEN IS SPECIFIED IN ADDR.                        *
;**************************************************************

WRWRD:  LCALL	CEHIGH		; WRITE VALUE IN DATAHI+LO INTO LOCATION
	MOV	A,ADDR		;  SPECIFIED IN ADDR
	RL	A		; JUSTIFY ADDRESS IN INSTRUCTION
	RL	A
	RL	A
	ORL     A,#WRITE	; MASK IN WRITE INSTRUCTION
	MOV	INST,A
	LCALL   OUTBYT		; SEND WRITE INSTRUCTION TO DUT
	MOV     A,DATAHI
	MOV	INST,A
	LCALL	OUTBYT          ; SEND IN UPPER BYTE OF DATA
	MOV	A,DATALO
	MOV	INST,A
	LCALL	OUTBYT		; SEND IN LOWER BYTE OF DATA
	LCALL	CELOW
	RET

;**********************************************************
; READ THE WORD ADDRESS SPECIFIED IN ADDR.  THE DATA WILL *
;  BE STORED IN DATALO AND DATAHI.                        *
;**********************************************************

RDWRD:  LCALL	CEHIGH		; READ THE ADDRESS SPECIFIED IN ADDR
	MOV	A,ADDR
	RL	A		; JUSTIFY ADDRESS TO READ
	RL	A
	RL	A
	ORL     A,#READ		; MASK IN READ INSTRUCTION
	MOV	INST,A
	LCALL	SEND7		; SEND IN 7 BITS OF READ INSTRUCTION
	SETB	DIO
	LCALL	CLOCK		; SEND EIGHTH CLOCK PULSE FOR READ INSTRUCTION
	MOV	COUNT,#10H	; PREPARE TO SHIFT IN 16 BITS
BITX:   MOV	C,DIO		; READ VALUE ON DO
NO1:	MOV     A,DATALO	; SHIFT DATA FROM DUT INTO DATALO
	RLC	A		;   AND DATAHI
	MOV	DATALO,A
	MOV	A,DATAHI
	RLC	A
	MOV	DATAHI,A
	LCALL	CLOCK		; SEND A CLOCK PULSE
	DJNZ	COUNT,BITX	; LOOP WHILE READING DATA
	LCALL	CELOW
	RET

;*****************************************************
; SEND DATA OUT TO THE PART.  THE DATA TO BE SENT IS *
;  LOCATED IN INST.                                  *
;*****************************************************

SEND7:	MOV	COUNT,#07H	; SHIFT OUT 7 BITS FOR READ INSTRUCTION
	SJMP	LOOPO
OUTBYT:	MOV	COUNT,#08H      ; PREPARE TO SHIFT OUT 8 BITS
LOOPO:	MOV	A,INST		; FIND DATA TO SHIFT OUT TO DI
	RLC	A
	MOV	INST,A
	MOV	DIO,C
	LCALL	CLOCK           ; SEND CLOCK SIGNAL
	DJNZ	COUNT,LOOPO     ; LOOP WHILE STILL SHIFTING OUT DATA
	RET

;****************
; BRING CE HIGH *
;****************

CEHIGH: SETB	CE		; BRING CE HIGH
	RET

;***************
; BRING CE LOW *
;***************

CELOW:  CLR	CE		; BRING CE LOW
	RET

;***********************
; ISSUE A CLOCK PULSE. *
;***********************

CLOCK:	SETB	SK		; SEND IN CLOCK PULSE (BRING SK HIGH)
	CLR	SK		; BRING SK LOW
	RET

	END
