;******************************************************************************
;*  This code was designed to demonstrate how the 2404 family of parts could  *
;*  be interface to the Z8 microcontroller.  The interface uses 2 lines       *
;*  from Port 2 (P2.0 and P2.1) to communicate.  Other IIC compatible parts   *
;*  can be added to the bus as long as they do not have $A as their device    *
;*  identifier.  The routines RDBYT and WRBYT are tailored specifically to the*
;*  2404 family.  The routines START, STOP, ACK, NACK, OUTBYT, and INBYT can  *
;*  be considered generic IIC routines.                                       *
;*                                                                            *
;*  The code shown demonstrates a 'random read' and 'byte write'.  The other  *
;*  modes of operation can be created by expanding upon these routines.       *
;*  Acknowledge polling is used to determine when the write cycle is complete.*
;*                                                                            *
;*  This code will work with all Xicor IIC compatible EEPROMs reguardless of  *
;*  their size.  As long as the address pins are configured correctly this    *
;*  code will not know the difference between a bus with a single 24C16 and a *
;*  bus with eight 2402s.                                                     *
;*                                                                            *
;*  The mainline of this program reads the data located at address 002DH and  *
;*  then writes that data back to address 0041H.  This program has been tested*
;*  using the 2402, 2404, 24C04, and 24C16.                                   *
;******************************************************************************

SDAHI	.EQU	#01H		; MASK USED TO FORCE SDA HIGH
SDALO	.EQU    #0FEH		;                    SDA LOW
SCLHI	.EQU    #02H		;                    SCL HIGH
SCLLO	.EQU    #0FDH		;                    SCL LOW

LOADDR	.EQU	R4              ; LOCATION FOR 2404 ADDRESS TO ACCESS LOW ORDER
HIADDR	.EQU	R5		;  HIGH ORDER
DATA1	.EQU	R6              ; LOCATION FOR 2404 DATA TRANSFERED
TEMP	.EQU	R7		; SCRATCH PAD BYTE
COUNT	.EQU	R8		; COUNTER VARIABLE
COUNT2	.EQU	R9		; COUNTER USED FOR ACK POLLING

;*************************************************************
; RESET TO LOCATION 000CH.  A8-15 ARE HI-Z AND HAVE PULL-UPS *
; SO THE EXTERNAL ADDRESS APPEARS TO BE 0FF0CH.              *
;*************************************************************

RESETV:	.ORG	0FF0CH  	; RESET VECTOR LOCATION
	JP	0FF0FH		; CHANGE PROGRAM COUNTER FROM 00XXH TO FFXXH

;***********************
; INITIALIZE PROCESSOR *
;***********************

INITL:	.ORG	0FF0FH		; INITIALIZE PROCESSOR
	LD	P0,#0FFH	; MAKE PORT0 (ADD 8-15) 0FFH
	LD	P01M,#96H	; MAKE PORT0 UPPER ORDER ADDRESS
	SRP	#00H		; REGISTER POINTER TO FIRST BANK
	LD	P2,#0FFH	; INITIALIZE PORT 2 TO 0FFH
	LD	P2M,#00H	; MAKE PORT 2 AN OUTPUT REGISTER
	LD	P3M,#00H	; MAKE PORT2 OUTPUTS OPEN DRAIN
	LD	SPL,#7FH	; SET STACK POINTER TO 007FH
	LD	SPH,#00H
	JP	BEGIN1		; JUMP TO START OF PROGRAM

;*********************************************
; BEGINNING OF CODE TO READ AND WRITE TO DUT *
;*********************************************

BEGIN1:	.ORG	0E000H
	LD	LOADDR,#2DH
	LD	HIADDR,#00H
	CALL	RDBYT           ; READ DATA FROM ADDRESS 002DH
	LD	LOADDR,#41H
	LD	HIADDR,#00H
	CALL	WRBYT		; WRITE DATA TO ADDRESS #0041H
	CALL	ACKPOL		; USE ACKNOWLEDGE POLLING
DONE:	JP	DONE            ; LOOP UNTIL RESET

;*********************************************************************
; READ A BYTE "RANDOM READ SEQUENCE".  THE ADDRESS TO READ IS STORED *
;  IN LOADDR AND HIADDR.  THE DATA FROM THE DUT IS STORED IN DATA1.  *
;*********************************************************************

RDBYT:	CALL	START           ; READ A BYTE FROM THE ADDRESS INDICATED
	LD	DATA1,HIADDR	; BUILD THE SLAVE ADDRESS (WRITE)
	RL	DATA1
	OR	DATA1,#0A0H
	CALL	OUTBYT          ; SEND SLAVE ADDRESS
	CALL	ACK             ; SEND ACKNOWLEDGE
	LD	DATA1,LOADDR
	CALL	OUTBYT          ; SEND WORD ADDRESS
	CALL	ACK             ; SEND ACKNOWLEDGE
	CALL	START           ; SEND START COMMAND
	LD	DATA1,HIADDR	; BUILD SLAVE ADDRESS (READ)
	RL	DATA1
	OR	DATA1,#0A1H
	CALL	OUTBYT          ; SEND SLAVE ADDRESS
	CALL	ACK             ; SEND ACKNOWLEDGE
	CALL	INBYT           ; READ DATA FROM 2404
	CALL	NACK            ; CLOCK WITHOUT ACKNOWLEDGE
	CALL	STOP            ; SEND STOP COMMAND
	RET

;**********************************************************************
; WRITE A BYTE "BYTE WRITE SEQUENCE".  THE ADDRESS TO WRITE IS STORED *
;  IN LOADDR AND HIADDR.  THE DATA TO WRITE IS STORED IN DATA1.       *
;**********************************************************************

WRBYT:	PUSH	DATA1           ; WRITE TO BYTE POINTED TO BY ADDR THE
	CALL	START           ; SEND START COMMAND
	LD	DATA1,HIADDR	; BYILD THE SLAVE ADDRESS
	RL	DATA1
	OR	DATA1,#0A0H
	CALL	OUTBYT          ; SEND SLAVE ADDRESS
	CALL	ACK             ; SEND ACKNOWLEDGE
	LD	DATA1,LOADDR
	CALL	OUTBYT          ; SEND WORD ADDRESS
	CALL	ACK             ; SEND ACKNOWLEDGE
	POP	DATA1
	CALL	OUTBYT          ; SEND WRITE DATA
	CALL	ACK             ; SEND ACKNOWLEDGE
	CALL	STOP            ; SEND STOP
	RET

;****************************************************************
; READ 8 BITS FROM THE DUT.  THE RESULTS ARE RETURNED IN DATA1. *
;****************************************************************

INBYT:	LD  	P2M,SDAHI       ; READ 8 BITS, MAKE SDA AN INPUT
	LD	COUNT,#08H
LOOPI:	CALL	CLOCK           ; CLOCK DATA
	RLC	DATA1		; STORE DUT DATA IN DATA1
	DJNZ	COUNT,LOOPI     ; LOOP UNTIL 8 BITS ARE READ
	LD	P2M,#00H	; MAKE SDA AN OUTPUT
	RET

;**********************************************************
; WRITE 8 BITS TO THE DUT.  THE DATA TO SEND IS IN DATA1. *
;**********************************************************

OUTBYT:	LD	COUNT,#08H      ; PREPARE TO SHIFT OUT 8 BITS
LOOPO:	RLC	DATA1		; ROTATE BIT TO SEND INTO CARRY FLAG
	JP	C,IS1		; SEND CARRY TO SDA
	AND	P2,SDALO	; WRITE 0 TO SDA
	JP	BITSET
IS1:	OR	P2,SDAHI	; WRITE 1 TO SDA
BITSET: CALL	CLOCK           ; SEND CLOCK SIGNAL TO DUT
	DJNZ	COUNT,LOOPO     ; LOOP UNTIL ALL 8 BITS HAVE BEEN SENT
	RET

;*********************************************************************
; PERFORM ACKNOWLEDGE POLLING TO DETERMINE WHEN THE WRITE CYCLE      *
;  COMPLETES.  UPON RETURN FROM THIS ROUTINE THE CARRY BIT INDICATES *
;  WHETHER THE DUT EVER ACKNOWLEDGED THE WRITE.  CARRY=0 PART        *
;  ACKNOWLEDGED, CARRY=1 NO ACKNOWLEDGE RECEIVED.                    *
;*********************************************************************

ACKPOL: LD	COUNT2,#80H	; MAX NUMBER OF TIMES TO CHECK THE PART
AKLOOP:	DJNZ	COUNT2,LOOK	; RETURN IF THE PART
	JP	OUTACK		;  NEVER ISSUES AN ACKNOWLEDGE
LOOK:	CALL	START		; SET UP TO READ A BYTE
	LD      DATA1,#0A0H	; BUILD SLAVE ADDRESS
	CALL	OUTBYT		; SEND SLAVE ADDRESS
	LD	P2M,SDAHI	; MAKE SDA AN INPUT
	CALL	NACK  		; SEE IF PART ACKNOWLEDGES
	LD	P2M,#00H	; MAKE SDA AN OUTPUT
	JP	C,AKLOOP	; LOOP IF NO ACKNOWLEDGE RECEIVED
OUTACK:	CALL	STOP		; ISSUE A STOP BEFORE RETURNING
	RET

;***********************
; ISSUE A STOP COMMAND *
;***********************

STOP:   AND	P2,SDALO        ; SEND STOP CONDITION TO DUT, SDA LOW
	OR  	P2,SCLHI	; SCL HIGH
	NOP
	NOP
	NOP
	NOP
	OR      P2,SDAHI	; SDA HIGH
	RET

;************************
; ISSUE A START COMMAND *
;************************

START:  OR   	P2,SDAHI	; SEND START CONDITION TO DUT, SDA HIGH
	OR	P2,SCLHI	; SCL HIGH
	NOP
	NOP
	NOP
	NOP
	AND	P2,SDALO	; SDA LOW
	NOP
	NOP
	NOP
	NOP
	AND	P2,SCLLO	; SCL LOW
	RET

;********************************************************
; ISSUE AN ACKNOWLEDGE.  THE ACK ROUTINE DOES NOT CHECK *
;  TO SEE IF THE DUT ACTUALLY ISSUES AN ACKNOWLEDGE.    *
;********************************************************

ACK:	AND	P2,SDALO        ; PERFORM AN ACKNOWLEDGE, SDA LOW
	CALL	CLOCK           ; GENERATE A CLOCK PULSE
	RET

;*********************************************
; CLOCK IN A 1 TO THE DUT.  THIS ROUTINE IS  *
;  CALLED WHEN A READ SEQUENCE HAS FINISHED. *
;*********************************************

NACK:   OR	P2,SDAHI	; CLOCK A 1 INTO THE DUT, SDA HIGH
	CALL	CLOCK           ; GENERATE A CLOCK PULSE
	RET

;******************************************************************
; ISSUE A CLOCK PULSE.  WHILE THE CLOCK IS HIGH THE VALUE ON THE  *
;  SDA LINE IS PLACED IN THE CARRY FLAG.  WHEN A READ IS TAKING   *
;  PLACE THE CARRY FLAG WILL INDICATE THE VALUE FROM THE DUT.     *
;******************************************************************

CLOCK:  NOP
	NOP
	OR	P2,SCLHI	; BRING SCL HIGH
	NOP
	NOP
	LD	TEMP,P2		; READ SDA
	RRC	TEMP		; SAVE SDA VALUE IN CARRY
	AND	P2,SCLLO	; SCL LOW
	RET

.END
