******************************************************************************
*  This code was designed to demonstrate how the Xicor 24C01 could           *
*  be interface to the 6803 microcontroller.  The interface uses 3 lines     *
*  from Port 1 (P1.0, P1.1, and P1.2) to communicate.                        *
*                                                                            *
*  The code shown demonstrates a 'random read' and 'byte write'.  The other  *
*  modes of operation can be created by expanding upon these routines.       *
*  Acknowledge polling is used to determine when the write cycle finishes.   *
******************************************************************************

SCLHI	EQU	$02              MASK TO GENERATE A 1 ON SCL
SCLLO	EQU	$FD                                 0    SCL
SDAHI	EQU	$01                                 1    SDA
SDALO	EQU	$FE                                 0    SDA
DMASK	EQU	$80              MASK TO DETERMINE WHAT STATE TO SHIFT INTO SDA

DDR1	EQU	$00              DATA DIRECTION REGISTER FOR PORT 1
PORT1	EQU	$02              ADDRESS FOR PORT 1
ADDR	EQU	$80              LOCATION FOR 2404 ADDRESS TO ACCESS
DATA	EQU	$82              LOCATION FOR 2404 DATA TRANSFERED
P1DATA	EQU	$83		 DATA TO BE SENT TO DUT
DD1DAT	EQU	$84              DATA TO BE STORED IN PORT 1 DIRECTION REGISTER
COUNT	EQU	$85		 COUNTER USED FOR ACK POLLING

****************************
* RESET VECTOR ENTRY POINT *
****************************

	ORG	$FFFE            RESET VECTOR TO PROGRAM ENTRY POINT
	FDB	$E000		 JUMP TO BEGINNING OF EXECUTABLE CODE

***********************
* PROGRAM ENTRY POINT *
***********************

	ORG	$E000            BEGINNING OF EXECUTABLE CODE

BEGIN:	LDS	#$00FF           INITIALIZE STACK POINTER
	LDAA	#$FF
	STAA	P1DATA           INITIALIZE PORT1 DATA VALUE
	STAA	DDR1             INITIALIZE PORT1 DIRECTION REGISTER
	STAA	DD1DAT           INITIALIZE PORT1 DIRECTION VALUE
	STAA	PORT1            INITIALIZE PORT1
	LDAA	#$2D
	STAA	ADDR
	JSR	RDBYT            READ DATA FROM ADDRESS 002DH
	LDAA	#$41
	STAA	ADDR

	STAA	DATA

	JSR	WRBYT
	JSR 	ACKPOL
	BRA	*                LOOP UNTIL RESET

**********************************************************************
* READ A BYTE "RANDOM READ SEQUENCE".  THE ADDRESS TO READ IS STORED *
*  IN ADDR.  THE DATA FROM THE DUT IS STORED IN DATA1.               *
**********************************************************************

RDBYT:	JSR	START            READ A BYTE FROM THE ADDRESS INDICATED
	LDAA	ADDR               IN 'ADDR'
	LSLA
	ORAA	#$01             BUILD SLAVE ADDRESS
	STAA	DATA
	JSR	OUTBYT           SEND SLAVE ADDRESS
	JSR	INBYT            READ DATA FROM 2404
	JSR	NACK             CLOCK WITHOUT ACKNOWLEDGE
	JSR	STOP             SEND STOP COMMAND
	RTS

***********************************************************************
* WRITE A BYTE "BYTE WRITE SEQUENCE".  THE ADDRESS TO WRITE IS STORED *
*  IN ADDR.  THE DATA TO WRITE IS STORED IN DATA1.                    *
***********************************************************************

WRBYT:	LDAA	DATA             WRITE TO BYTE POINTED TO BY ADDR THE
	PSHA
	JSR	START            SEND START COMMAND
	LDAA	ADDR
	LSLA
	STAA	DATA
	JSR	OUTBYT           SEND SLAVE ADDRESS
	JSR	ACK              SEND ACKNOWLEDGE
	PULA
	STAA	DATA
	JSR	OUTBYT           SEND WRITE DATA
	JSR	ACK              SEND ACKNOWLEDGE
	JSR	STOP             SEND STOP
	RTS

*****************************************************************
* READ 8 BITS FROM THE DUT.  THE RESULTS ARE RETURNED IN DATA1. *
*****************************************************************

INBYT:	LDAA	DD1DAT
	ANDA	#$FE
	STAA	DDR1
	STAA	DD1DAT
	JSR 	CLOCK
	LDX	#$0008           PREPARE TO SHIFT IN 8 BITS
	LDAB	#$00
LOOPI:	JSR	CLOCK            CLOCK DATA
	LSRA
	ROLB
	DEX
	BNE	LOOPI            LOOP UNTIL 8 BITS ARE READ
	STAB	DATA
	LDAA	DD1DAT
	ORAA	#$01
	STAA	DDR1
	STAA	DD1DAT
	RTS

***********************************************************
* WRITE 8 BITS TO THE DUT.  THE DATA TO SEND IS IN DATA1. *
***********************************************************

OUTBYT:	LDX	#$0008           PREPARE TO SHIFT OUT 8 BITS
	LDAA	DATA
LOOPO:	LDAB	P1DATA
	ANDB	#SDALO
	ANDA	#DMASK            IS THE DATA TO BE SHIFTED A 1 OR A 0
	BEQ	IS0              JUMP IF DATA SHOULD BE 0
	ORAB	#SDAHI            MAKE DATA A 1
IS0:	STAB	PORT1            PUT DATA ON SDA
	STAB	P1DATA
	JSR	CLOCK            SEND CLOCK SIGNAL
	LDAA	DATA
	ASLA
	STAA	DATA
	DEX
	BNE	LOOPO            LOOP UNTIL ALL 8 BITS HAVE BEEN SENT
	RTS

************************************************************************
* PERFORM ACKNOWLEDGE POLLING TO DETERMINE WHEN THE WRITE CYCLE        *
* COMPLETES.  UPON RETURN FROM THIS ROUTINE THE A REGISTER INDICATES   *
* WHETHER THE DUT EVER ACKNOWLEDGED THE WRITE.  A=0 PART ACKNOWLEDGED, *
* A=1 NO ACKNOWLEDGE RECEIVED.                                         *
************************************************************************

ACKPOL:	LDAA	#$080		 MAX NUMBER OF TIMES TO CHECK THE PART
	STAA	COUNT
AKLOOP:	DEC	COUNT            RETURN IF THE PART
	BEQ	OUTACK		  NEVER ISSUES AN ACKNOWLEDGE
	JSR	START            SEND START COMMAND
	LDAA	#$00
	STAA	DATA
	JSR	OUTBYT           SEND SLAVE ADDRESS
	JSR	NACK             SEE IF PART ACKNOWLEDGES
	BNE	AKLOOP		 LOOP IF NO ACKNOWLEDGE
OUTACK: PSHA
	JSR	START		 SEND START
	JSR	STOP             SEND STOP
	PULA
	RTS

************************
* ISSUE A STOP COMMAND *
************************

STOP:	LDAA	P1DATA           PROVIDE A STOP CONDITION
	ANDA	#SDALO
	STAA	PORT1            MAKE SURE SDA IS LOW
	ORAA	#SCLHI
	STAA	PORT1            BRING SCL HIGH
	ORAA	#SDAHI
	STAA	PORT1            BRING SDA HIGH
	STAA	P1DATA
	RTS

*************************
* ISSUE A START COMMAND *
*************************

START:	LDAA	P1DATA           MAKE SURE THAT SDA IS HIGH
	ORAA	#SDAHI
	STAA	PORT1
	ORAA	#SCLHI           MAKE SURE THAT SCL IS HIGH
	STAA	PORT1
	ANDA	#SDALO
	STAA	PORT1            BRING SDA LOW
	ANDA	#SCLLO
	STAA	PORT1            BRING SCL LOW
	STAA	P1DATA
	RTS

*********************************************************
* ISSUE AN ACKNOWLEDGE.  THE ACK ROUTINE DOES NOT CHECK *
*  TO SEE IF THE DUT ACTUALLY ISSUES AN ACKNOWLEDGE.    *
*********************************************************

ACK:	LDAA	P1DATA           PERFORM AN 'ACKNOWLEDGE'
	ANDA	#SDALO
	STAA	PORT1            BRING SDA LOW
	STAA	P1DATA
	JSR	CLOCK            GENERATE A CLOCK PULSE
	RTS

**********************************************
* CLOCK IN A 1 TO THE DUT.  THIS ROUTINE IS  *
*  CALLED WHEN A READ SEQUENCE HAS FINISHED. *
**********************************************

NACK:	LDAA	P1DATA           PERFORM AN 'ACK' WITH SDA HIGH
	ORAA	#SDAHI
	STAA	PORT1            BRING SDA HIGH
	STAA	P1DATA
	JSR	CLOCK            GENERATE A CLOCK PULSE
	RTS

******************************************************************
* ISSUE A CLOCK PULSE.  WHILE THE CLOCK IS HIGH THE VALUE ON THE *
*  SDA LINE IS PLACED IN THE CARRY FLAG.  WHEN A READ IS TAKING  *
*  PLACE THE CARRY FLAG WILL INDICATE THE VALUE FROM THE DUT.    *
******************************************************************

CLOCK:	LDAA	P1DATA           PROVIDE A CLOCK PULSE ON SCL
	ORAA	#SCLHI
	STAA	PORT1            BRING SCL HIGH
	LDAA	PORT1            READ SDA WHILE SCL IS HIGH
	PSHA
	ANDA	#SCLLO
	STAA	PORT1            BRING SCL LOW
	STAA	P1DATA
	PULA
	ANDA	#$01             SDA VALUE IS IN LOWER BIT OF A REG
	RTS
