/*
 * LWRAN.H -- LightWave Rendering and ANimation plug-in server description.
 *
 * written by Stuart Ferguson
 * last revision  12/21/94
 */
#ifndef LW_RAN_H
#define LW_RAN_H

#include <lwbase.h>


	/*
	 * ================================
	 * COMMON DEFINITIONS
	 *
	 * These types and definitions are common to the various plugins
	 * and global accessors.
	 */


/*
 * Coordinate and Range Scales
 *
 * There are several conventions used to interpret different types of values
 * within LightWave and throughout this external interface.
 *
 * positions	Positions are always given as an array of three doubles which
 *		are the X, Y and Z coordinates (respectively) of a position
 *		with respect to some known coordinate system.  This system is
 *		always listed along with the vector (usually object or world).
 *
 * directions	Direction vectors, such as normals, are also an X Y Z array of
 *		doubles, but are normalized to be unit vectors.  The coordinate
 *		system for these is also always listed.
 *
 * rotations	Rotations are arrays of three doubles representing the Euler
 *		angles (heading, pitch and bank, respectively) of an item in
 *		some coordinate system.  Angles are represented in radians.
 *
 * colors	Colors are given as an array of three doubles giving the
 *		intensities of the red, green and blue componenets of the color,
 *		respectively.  The values are scaled so that 0.0 is the minimum
 *		intensity and 1.0 is the maximum, although out-of-range values
 *		are allowed.
 *
 * percentages	Values which are represented on the LightWave user interface as
 *		percentages are represented internally as doubles scaled from
 *		0.0 to 1.0.  A luminosity of 65.7%, for example, would be held
 *		internally as the floating point value 0.657.
 *
 * Floating point values with a nominal range of 0.0 to 1.0 will sometimes be
 * converted to single-byte values for storing in image buffers.  Colors and
 * alphas are converted for image output and other values are converted for
 * internal use.  In this case the value is clipped to be strictly within the
 * 0.0 to 1.0 range and is then scaled and converted to a BufferValue type so
 * that 0.0 is 0 and 1.0 is 255.
 */
typedef unsigned char	  BufferValue;


/*
 * Items and Properties
 *
 * A LightWave item is anthing which can be keyframed in the layout interface.
 * All objects, lights, bones and cameras in LightWave are items and have a
 * unique LWItemID value.  Item classifications are given by LWItemType codes.
 */
typedef void *		 LWItemID;
#define LWITEM_NULL	 ((LWItemID) 0)

typedef int		 LWItemType;
#define LWI_OBJECT	 0
#define LWI_LIGHT	 1
#define LWI_CAMERA	 2
#define LWI_BONE	 3

/*
 * All items have a set of vector parameters which servers can read (and
 * sometimes write) using these property codes.
 */
typedef int		 LWItemParam;
#define LWIP_POSITION	 1	/* position (parent's coords) */
#define LWIP_RIGHT	 2	/* item +X direction (world coords) */
#define LWIP_UP		 3	/* item +Y direction (world coords) */
#define LWIP_FORWARD	 4	/* item +Z direction (world coords) */
#define LWIP_ROTATION	 5	/* rotation (parent's coords) */
#define LWIP_SCALING	 6	/* scaling relative to parent */
#define LWIP_PIVOT	 7	/* pivot point (object coords) */
#define LWIP_W_POSITION	 8	/* position (world coords) */
#define LWIP_W_RIGHT	 9	/* world +X direction (item coords) */
#define LWIP_W_UP	 10	/* world +Y direction (item coords) */
#define LWIP_W_FORWARD	 11	/* world +Z direction (item coords) */


/*
 * Time
 *
 * Time values in LightWave are given in two ways.  A frame number is
 * the index of a single image (typically the current image) in the output
 * sequence of still images that make up the animation.  A time value
 * is the precise instant of an event in seconds.  Assuming a scene
 * rendered at 30 frames per second and without motion blur (or with a blur
 * length of zero), frame N is a snapshot of the animation at a time in
 * seconds of N/30.  If there is motion blur, then some events in frame
 * N will be from times slightly before N/30 seconds, and if the motion
 * blur length is greater than 100%, then some events may even overlap
 * with the times of events in previous frames.
 */
typedef int		 LWFrame;
typedef double		 LWTime;


/*
 * Errors
 *
 * Server functions return errors to LightWave by returning a string pointer.
 * A null string pointer indicates no error, and a non-null pointer points
 * to an error string.  The string will be displayed for the user and, except
 * where otherwise indicated, the user will have the option to ignore the
 * error and continue with the operation.
 */
typedef const char *	 LWError;


/*
 * Instances and Handlers
 *
 * Most LightWave plugins are "handlers" which manage "instances."  An
 * instance is a specific collection of user settings for a texture, image
 * filter, etc., which persist across sessions by being stored in scene and
 * object files.  A `LWInstance' is any longword value which identifies a
 * specific instance for a specific server, usually a pointer to allocated
 * memory.
 */
typedef void *		 LWInstance;

/*
 * Instances have to load and save their data to and from ASCII scene files
 * and binary object files, and sometimes both, so the data read/write
 * mechanism provides servers with functions to read and write data in both
 * these formats.  The format of the file is given by `ioMode' code, either
 * OBJECT or SCENE.  The `read' function reads bytes from the source and
 * returns the number of bytes read.  The `write' function writes bytes to
 * the output and tracks any errors internally.
 *
 * If the mode is OBJECT, the format is binary and no scene-specific
 * information should be stored.  The read and write functions deal in raw
 * bytes which can have any value from 0 to 255.  The read functions reads
 * the number of bytes requested or fewer if the end of input is reached.
 *
 * If the mode is SCENE, the format is ASCII and bytes stored must be in
 * the extended ASCII range of 32 to 255.  Values outside this range are
 * ignored or undefined.  The read/write functions in this case deal with
 * lines.  The write function writes a line at a time and looks for a null
 * terminator in the input rather than the length.  The read function can
 * read partial lines if a length less then or equal to the total line
 * length is requested.  If the length is greater than the remaining line
 * length, the length is returned and the buffer is null-terminated.  The
 * read function returns -1 for the actual end of input, since a read
 * length of zero is valid for a blank line.
 *
 * Plugin clients which write instance data must do their own versioning so
 * they can read old forms of their own data, and their own bit twiddling
 * to read and write binary data on machines with different byte order and
 * floating point formats.
 */
#define LWIO_OBJECT	 0
#define LWIO_SCENE	 1

typedef struct st_LWLoadState {
	int		  ioMode;
	void		 *readData;
	int		(*read) (void *readData, char *buf, int len);
} LWLoadState;

typedef struct st_LWSaveState {
	int		  ioMode;
	void		 *writeData;
	void		(*write) (void *writeData, char *buf, int len);
} LWSaveState;

/*
 * A server manages its instances by providing LightWave with functions to
 * create, destroy, load and save them.  The server activation function gets
 * a handler structure which it initializes with the standard instance
 * handler functions listed here, plus whatever else is required by the
 * specific class of plugin.
 *
 * create	- create a default or uninitialized instance.  Any failure
 *		  should return a null pointer and optionally set the error
 *		  value.
 * destroy	- dispose of an instance.
 * copy		- copy the contents of the "from" instance to the "to"
 *		  instance.
 * load		- read an instance description from a file into an already
 *		  created instance.
 * save		- write an instance description to a file.
 *
 * The create function will always be used in conjunction with the load
 * function or the interface function.  A new instance must always be
 * initialized using one of these two methods before it is used.
 */
typedef struct LWInstHandler {
	LWInstance	(*create) (LWError *);
	void		(*destroy) (LWInstance);
	LWError		(*copy) (LWInstance from, LWInstance to);
	LWError		(*load) (LWInstance, const LWLoadState *);
	LWError		(*save) (LWInstance, const LWSaveState *);
} LWInstHandler;

/*
 * There is often another server associated with a handler and that is the
 * "Interface" server.  The activation function for the interface server
 * is called with a LWInstance as its local data.  The server will then
 * allow the user to edit the instance description and return.  The
 * interface server is just another function that operates on instances of
 * a specific type, but it is separate from the other instance handler
 * functions for two reasons.  The first is that the user interface code
 * is frequently the largest and least often used part of a handler, so it
 * makes sense to allow it to be loaded separately only when needed.  The
 * second is to permit plugin authors to easily make "render-only" versions
 * of their plugin servers for rendering accelerators or packaging bundles.
 *
 * For example, if the plugin type was "XXX," there would be two classes of
 * server, "XXXHandler" and "XXXInterface."  Then for a specific server of
 * the XXX type, called "MyXXX," there would be a MyXXX defined for the
 * XXXHandler class which would provide all the normal handler functions,
 * and there would also be a MyXXX server of the XXXInterface class which
 * would perform the user interface.
 */





	/*
	 * ================================
	 * SERVER CLASSES
	 *
	 * The server classes are given by the class name(s) and describe
	 * the data types and structures used in its operation.
	 */


/*
 * -----------------
 * Image Post Processing -- "ImageFilterHandler" & "ImageFilterInterface"
 *
 * Image post processing (filtering) effects are applied to the final
 * rendered image after all the antialiasing and motion blur passes are
 * complete.  The operation should apply its effect by modifying the red,
 * green, blue and alpha values of the final image.  The server can compute
 * these by looking at the contents of a potentially large set of full-image
 * buffers, given by the LWBUF codes below.  The RED, GREEN, BLUE and ALPHA
 * buffers would be the outputs if the image operator did nothing.  SPECIAL
 * is a value assigned to specific surfaces for only this filter.  The
 * remaining buffers are intermediate computed values which can be useful
 * for some special effects.
 */
#define LWBUF_SPECIAL	 0	/* server-specific value */

#define LWBUF_LUMINOUS	 1	/* pixel values before shading */
#define LWBUF_DIFFUSE	 2
#define LWBUF_SPECULAR	 3
#define LWBUF_MIRROR	 4
#define LWBUF_TRANS	 5
#define LWBUF_RAW_RED	 6
#define LWBUF_RAW_GREEN	 7
#define LWBUF_RAW_BLUE	 8

#define LWBUF_SHADING	 9	/* raw shading applied to surfaces */
#define LWBUF_SHADOW	 10	/* amount of shadow at each pixel */
#define LWBUF_GEOMETRY	 11	/* cosine of eye vector angle */
#define LWBUF_DEPTH	 12	/* depth of pixel in image (float) */

#define LWBUF_RED	 32	/* RGBA of the rendered image. */
#define LWBUF_GREEN	 33	/*	These are always provided */
#define LWBUF_BLUE	 34
#define LWBUF_ALPHA	 35

/*
 * The contents of the buffers must be accessed through the `bufLine' or
 * `fltLine' functions which return pointers to a line of the buffer of the
 * given type.  For y=0, the top line of the buffer is returned; for y=1
 * the second to the top line, etc.  `bufLine' returns lines from
 * byte-encoded buffers and `fltLine' returns lines from float-encoded
 * buffers (currently only DEPTH).  Invalid type codes return null pointers.
 * The output buffers must be set using `setRGB' and `setAlpha' which set
 * the final value at a location.  The source buffers do not change as the
 * output buffers are modified.  The width and height of all buffers is the
 * same, and the time values give a range if there is any motion blur in the
 * image.
 */
typedef struct st_FilterAccess {
	int		  width, height;
	LWFrame		  frame;
	LWTime		  start, end;
	BufferValue *	(*bufLine) (int type, int y);
	float *		(*fltLine) (int type, int y);
	void		(*setRGB)   (int x, int y, BufferValue[3]);
	void		(*setAlpha) (int x, int y, BufferValue);
} FilterAccess;

/*
 * The activation function for an image filter gets passed a blank handler
 * structure as its local data which the server must fill in.  In addition
 * to the normal instance functions, it must also provide a `process'
 * function and `flags' function.
 *
 * process	- called with instance data and buffer access functions to
 *		  image process a rendered frame.
 * flags	- returns a set of bits representing the buffers it wants
 *		  at processing time, where the bit numbers are the LWBUF
 *		  values above.  Only buffers 0-12 need to be specified
 *		  this way since the R, G, B and Alpha buffers are always
 *		  provided.  Undefined bits should be clear by default.
 */
typedef struct st_ImageFilterHandler {
	LWInstHandler	  inst;
	void		(*process) (LWInstance, const FilterAccess *);
	unsigned int	(*flags) (LWInstance);
} ImageFilterHandler;



/*
 * -----------------
 * Procedural Texture -- "ShaderHandler" & "ShaderInterface"
 *
 * A shader is called to modify the rendering attributes of a pixel as
 * it is being rendered.  Since it is called on a per-pixel basis, the
 * interface is designed for speed.
 *
 * The access structure for shader evaluation gets some values which are
 * read-only and some which are meant to be modified.  The read-only
 * values describe the geometry of the pixel being shaded.  The read-write
 * values describe the current shading properties of this pixel and should
 * be modified in place to affect the look of the shader.  Since shaders
 * may be layered, these properties may be altered many more times before
 * final rendering.  If the shader does bump mapping, it does not not need
 * to renormalize the surface normal after perturbation.
 *
 * Special functions are provided to shaders which are not available in
 * any other context.  The `illuminate' function returns the light ray
 * (color and direction) hitting the given position from the given light
 * at the current instant.  The return value is zero if the light does not
 * illuminate the given world coordinate position at all.  The color
 * includes effects from shadows (if any), falloff, spotlight cones and
 * transparent objects between the light and the point.  The `rayTrace'
 * function may be called to trace a ray from the current location in a
 * given direction (in world coordinates).  The return value is the length
 * of the ray (or -1.0 if infinite) and the color coming from that
 * direction.  The direction used is the outgoing direction and does not
 * need to be normalized.
 */
typedef struct st_ShaderAccess {
	/*
	 * Read-only geometry information.
	 */
	int		 sx, sy;	/* pixel location (pixels from
					 * upper left-hand corner) */
	double		 oPos[3];	/* pixel location (object coords) */
	double		 wPos[3];	/* pixel location (world coords) */
	double		 spotSize;	/* approx. pixel size (world coords) */
	double		 raySource[3];	/* incoming ray origin (world coords) */
	double		 rayLength;	/* incoming ray length */
	double		 cosine;	/* cosine of ray and surface normal */
	double		 oXfrm[9];	/* object to world transform matrix */
	double		 wXfrm[9];	/* world to object transform matrix */

	/*
	 * Read-modify-write shading information.
	 */
	double		 wNorm[3];	/* surface normal (world coords) */
	double		 color[3];	/* surface color */
	double		 luminous;	/* luminosity */
	double		 diffuse;	/* diffuse level */
	double		 specular;	/* specular level */
	double		 mirror;	/* reflectivity */
	double		 transparency;
	double		 eta;		/* refractive index */
	double		 roughness;	/* 1 / glossiness */

	/*
	 * Special functions.
	 */
	int	       (*illuminate) (LWItemID, const double position[3],
				      double direction[3], double color[3]);
	double	       (*rayTrace) (const double direction[3], double color[3]);
} ShaderAccess;

/*
 * A shader instance may store its data in an object (in the case of a
 * surface texture) or in a scene (in the case of a clip map) so the
 * save/load functions should be prepared to deal with both cases.
 *
 * init		- called at the start of rendering a sequence of frames.
 * cleanup	- called when current sequence is complete.
 * newTime	- called at the start of each new time within the
 *		  current sequence.
 * evaluate	- called to compute the shading of each affected pixel
 *		  within the current time.
 * flags	- returns a word containing status bits for the instance.
 *		  Undefined flag bits should be clear by default.
 */
typedef struct st_ShaderHandler {
	LWInstHandler	  inst;
	LWError		(*init) (LWInstance);
	void		(*cleanup) (LWInstance);
	LWError		(*newTime) (LWInstance, LWFrame, LWTime);
	void		(*evaluate) (LWInstance, const ShaderAccess *);
	unsigned int	(*flags) (LWInstance);
} ShaderHandler;

/*
 * The first nine LWSHF bits should be set only if the shader instance is
 * going to modify the particular attribute.  Alternately, all those bits
 * can be clear and the shader can set the TRUECOLOR bit, in which case
 * the pixel is not shaded at all and its color comes directly from the
 * color in the ShaderAccess struct.  RAYTRACE must be set if the shader
 * intends to use the `rayTrace' function.
 */
#define LWSHF_NORMAL	(1<<0)
#define LWSHF_COLOR	(1<<1)
#define LWSHF_LUMINOUS	(1<<2)
#define LWSHF_DIFFUSE	(1<<3)
#define LWSHF_SPECULAR	(1<<4)
#define LWSHF_MIRROR	(1<<5)
#define LWSHF_TRANSP	(1<<6)
#define LWSHF_ETA	(1<<7)
#define LWSHF_ROUGH	(1<<8)
#define LWSHF_TRUECOLOR	(1<<9)
#define LWSHF_RAYTRACE	(1<<10)


/*
 * -----------------
 * Procedural Displacement Map -- "DisplacementHandler"
 *				& "DisplacementInterface"
 *
 * A displacement map is called before rendering to modify the geometry
 * of an object.  This is done not only during rendering but also during
 * interactive previewing in the Layout window.  This means that a server
 * should always be prepared to process a displacement instance at any
 * time.
 *
 * The access structure for a displacement map gets the position of the
 * point to displace in two ways.  The first is the point relative to the
 * object, and is read-only.  The second, the `source' position, is the
 * one to be altered.  If this is not a world-coordinate displacement,
 * then the source coordinates are the in the object coordinate system but
 * have been already displaced by any morphing or boning applied to the
 * object, and may differ from the object coordinates.  If the displacment
 * handler has specified world coordinates (a flag bit returned by the
 * `flags' function), then the source coordinates are morphed, boned and
 * transformed by object motion (i.e. they are world coordinates).
 */
typedef struct st_DisplacementAccess {
	double		 oPos[3];	/* vertex position (object coords) */
	double		 source[3];	/* starting coordinates for displacement */
} DisplacementAccess;

/*
 * The handler functions for a displacement map are the same as a shader
 * except for the lack of `init' and `cleanup' functions.  The `newTime'
 * function also has a parameter for the ID of the object being affected
 * by the displacement.  The LWDMF_WORLD bit should be set in the `flags'
 * return value if the displacement will take place in world coordinates.
 */
typedef struct st_DisplacementHandler {
	LWInstHandler	  inst;
	LWError		(*newTime) (LWInstance, LWItemID, LWFrame, LWTime);
	void		(*evaluate) (LWInstance, const DisplacementAccess *);
	unsigned int	(*flags) (LWInstance);
} DisplacementHandler;

#define LWDMF_WORLD	(1<<0)


/*
 * -----------------
 * Procedural Item Animation -- "ItemMotionHandler" & "ItemMotionInterface"
 *
 * Animation behavior can be applied to any item in a scene which can be
 * keyframed.  After the keyframe position of the item is computed, the
 * item motion server can alter the keyframed motion or replace it with a
 * completely different one.  Motions will be evaluated both during
 * rendering and while interactively laying out a scene.
 *
 * At each time instant and for each affected item, the motion evaluation
 * function will be called with an access structure holding the ID of the
 * item and the time instant for which the motion should be computed.  The
 * server can query parameters for the item at the given time using the
 * `getParam' function and a LWItemParam code.  Some subset of these
 * values may then be modified using the `setParam' function.
 *
 * Procedural motions are not currently allowed to interact.  If a motion
 * evaluation function attempts to read out the position of another object
 * which is affected by a procedural motion, only the values of the
 * keyframed motion will be returned.
 */
typedef struct st_ItemMotionAccess {
	LWItemID	  item;
	LWFrame		  frame;
	LWTime		  time;
	void		(*getParam) (LWItemParam, double vector[3]);
	void		(*setParam) (LWItemParam, const double vector[3]);
} ItemMotionAccess;

/*
 * The server should fill in the function pointers in the handler struct.
 * The `evaluate' function is called for each item at each timestep to
 * compute its motion.
 */
typedef struct st_ItemMotionHandler {
	LWInstHandler	  inst;
	void		(*evaluate) (LWInstance, const ItemMotionAccess *);
} ItemMotionHandler;



/*
 * -----------------
 * Procedural Object Replacement -- "ObjReplacementHandler"
 *				  & "ObjReplacementInterface"
 *
 * Another type of animation is allowed which can replace the entire
 * object geometry at every single step.  The step evaluation function
 * is called with the frame, time and type values for the currently
 * loaded geometry and for the next step being evaluated.  The server
 * only has to set the `newFilename' value if new geometry should be
 * loaded.  If the currently loaded geometry can be used for the new
 * frame and time, then the new filename can be set to null.
 *
 * Filenames are used instead of direct mesh replacement for generality.
 * An object replacement server could use a series of prebuilt objects,
 * like character heads for example, to do expressions or lip-syncing by
 * providing the name of the correct head at each step.  Some animation
 * could be done very efficiently using a combination of object
 * replacement and object import servers.  The replacement server could
 * write a brief description file for the parameters of a timestep
 * (positions and sizes of metaballs, for example) which the object
 * import server could then convert into a complete mesh while loading.
 */
typedef struct st_ObjReplacementAccess {
	LWItemID	 objectID;
	LWFrame		 curFrame, newFrame;
	LWTime		 curTime,  newTime;
	int		 curType,  newType;
	const char	*curFilename;
	const char	*newFilename;
} ObjReplacementAccess;

/*
 * An object replacement server can provide different geometry for
 * interactive previewing and actual rendering by examining the `type'
 * value for a step.  NONE is only set for `curType' when there is no
 * geometry loaded at all for this item.
 */
#define OBJREP_NONE	 0
#define OBJREP_PREVIEW	 1
#define OBJREP_RENDER	 2

/*
 * The server should fill in the function pointers in the handler struct.
 * The `evaluate' function is called for each affected object at each
 * timestep to get new geometry.
 */
typedef struct st_ObjReplacementHandler {
	LWInstHandler	  inst;
	void		(*evaluate) (LWInstance, const ObjReplacementAccess *);
} ObjReplacementHandler;


/*
 * -----------------
 * Frame Buffers -- "FrameBufferHandler" & "FrameBufferInterface"
 *
 * LightWave uses a frame buffer to display the output of rendering as
 * each frame is completed.  This is for the user to view, so the a
 * frame buffer should also be able to pause waiting for user input.
 *
 * A frame buffer is an instance, but it may be very limited.  The built-
 * in frame buffers have no UI and no stored state.
 *
 * open		- open display at the given size.
 * close	- close display and end display transations.
 * begin	- start a new frame.
 * write	- write a new line of RGB and alpha data to the
 *		  framebuffer.  Lines always come from top to bottom and
 *		  there are always enough to fill the width and height of
 *		  the requested display.
 * pause	- display the buffer to the user and wait for their signal
 *		  to continue before returning.
 *
 * The sequence of calls for rendering to the frame buffer can be
 * visualized as a regular expression:
 *	open, (begin, (write)H, pause?)*, close
 *
 * Any number of frames may be displayed in a session.  Write will always
 * be called for all the lines in the image and pause is optional.
 */
typedef struct st_FrameBufferHandler {
	LWInstHandler	  inst;
	LWError		(*open) (LWInstance, int w, int h);
	void		(*close) (LWInstance);
	LWError		(*begin) (LWInstance);
	LWError		(*write) (LWInstance, const BufferValue *R,
					      const BufferValue *G,
					      const BufferValue *B,
					      const BufferValue *alpha);
	void		(*pause) (LWInstance);
} FrameBufferHandler;



/*
 * -----------------
 * Animation Output -- "AnimSaverHandler" & "AnimSaverInterface"
 *
 * Animations are written out using a nearly identical scheme as that for
 * framebuffers, except that there is no `pause' function and in addition
 * to the image size, LightWave will also pass a filename.
 */
typedef struct st_AnimSaverHandler {
	LWInstHandler	  inst;
	LWError		(*open) (LWInstance, int w, int h, const char *filename);
	void		(*close) (LWInstance);
	LWError		(*begin) (LWInstance);
	LWError		(*write) (LWInstance, const BufferValue *R,
					      const BufferValue *G,
					      const BufferValue *B,
					      const BufferValue *alpha);
} AnimSaverHandler;



/*
 * -----------------
 * Scene Conversion -- "SceneConverter"
 *
 * The import of foreign scene formats is accomplished with scene converters.
 * When the user selects a file to load as a scene, LightWave first attempts
 * to load it directly as an LWSC format file.  It it cannot, it will pass
 * the filename to each scene converter in sequence.  If the scene converter
 * can read the file it will write a temporary LWSC file and pass the name
 * back to LightWave as `tmpFile.'  The file will be loaded and the `delete'
 * function will be called to dispose of the temp scene file and any related
 * temp files.  If the scene converter does not recognize the format it should
 * just return and LightWave will try another converter.  If the converter can
 * recognize the format but cannot convert the file for some reason it should
 * leave `tmpScene' as null but set the `readFailure' for the error
 * encountered.  LightWave will assume that the file cannot be converted and
 * display the message.
 */
typedef struct st_SceneConverter {
	const char	 *filename;
	LWError		 *readFailure;
	const char	 *tmpScene;
	void		(*delete) (const char *tmpScene);
} SceneConverter;





	/*
	 * ================================
	 * GLOBAL DATA
	 *
	 * This section contains descriptions of the global data pointers
	 * which can be accessed from LightWave's global function.  The ID
	 * string for each global is given in quotes.
	 */


/*
 * -----------------
 * Item Information -- "LW Item Info"
 *
 * Plugin clients can traverse the entire set of items for the current scene
 * and get information about all of them using these interfaces.
 *
 * first	- get the ID of the first item of a given type.  If type is
 *		  LWI_BONE, the second argument is the ID of the boned object.
 *		  If there are no items of this type this returns LWITEM_NULL.
 * next		- get the next item of the same type as the argument.  If there
 *		  are no more, this returns LWITEM_NULL.
 * firstChild	- get the first child item of the parent item.  It returns
 *		  LWITEM_NULL if none.
 * nextChild	- get the next child item given a parent item and the previous
 *		  child.  It returns LWITEM_NULL if that was the last one.
 * parent	- get the item's parent, if any, and LWITEM_NULL if none.
 * target	- get the item's target, if any, and LWITEM_NULL if none.
 * parent	- get the item's goal, if any, and LWITEM_NULL if none.
 * type		- get the type of an arbitrary item.
 * name		- get the name of an item as it appears to the user.
 * param	- retrieve vector parameters from an item using a LWItemParam
 *		  code to identify the parameter desired.  The value is written
 *		  to the vector array for the given time.
 * limits	- retrieve upper and lower bounds on vector parameters.  LWVECF
 *		  flag bits are returned to indicate which of the three vector
 *		  indicies contain valid limits.
 */
typedef struct st_LWItemInfo {
	LWItemID	(*first)  (LWItemType, LWItemID);
	LWItemID	(*next)   (LWItemID);
	LWItemID	(*firstChild) (LWItemID parent);
	LWItemID	(*nextChild) (LWItemID parent, LWItemID prevChild);
	LWItemID	(*parent) (LWItemID);
	LWItemID	(*target) (LWItemID);
	LWItemID	(*goal)   (LWItemID);
	LWItemType	(*type)   (LWItemID);
	const char *	(*name)   (LWItemID);
	void		(*param)  (LWItemID, LWItemParam, LWTime,
				   double vector[3]);
	unsigned int	(*limits) (LWItemID, LWItemParam,
				   double min[3], double max[3]);
} LWItemInfo;

#define LWVECF_0	(1<<0)
#define LWVECF_1	(1<<1)
#define LWVECF_2	(1<<2)


/*
 * -----------------
 * Object Information -- "LW Object Info"
 *
 * Given the item ID for an object, a client can get some object-specific
 * information.  `shadowOpts' returns bits for shadow options.
 */
typedef struct st_LWObjectInfo {
	const char *	(*filename) (LWItemID);
	int		(*numPoints) (LWItemID);
	int		(*numPolygons) (LWItemID);
	unsigned int	(*shadowOpts) (LWItemID);
	double		(*dissolve) (LWItemID, LWTime);
} LWObjectInfo;

#define LWOSHAD_SELF	 (1<<0)
#define LWOSHAD_CAST	 (1<<1)
#define LWOSHAD_RECEIVE	 (1<<2)



/*
 * -----------------
 * Bone Information -- "LW Bone Info"
 *
 * Given the item ID for a bone, a client can get some bone-specific
 * information.
 */
typedef struct st_LWBoneInfo {
	unsigned int	(*flags) (LWItemID);
	void		(*restParam) (LWItemID, LWItemParam, double vector[3]);
	double		(*restLength) (LWItemID);
	void		(*limits) (LWItemID, double *inner, double *outer);
} LWBoneInfo;

#define LWBONEF_ACTIVE		(1<<0)
#define LWBONEF_LIMITEDRANGE	(1<<1)


/*
 * -----------------
 * Light Information -- "LW Light Info"
 *
 * Given the item ID for a light, a client can get some light-specific
 * information.
 */
typedef struct st_LWLightInfo {
	void		(*ambient) (LWTime, double color[3]);
	int		(*type) (LWItemID);
	void		(*color) (LWItemID, LWTime, double color[3]);
	int		(*shadowType) (LWItemID);
	void		(*coneAngles) (LWItemID, double *radius, double *edge);
} LWLightInfo;

#define LWLIGHT_DISTANT	 0
#define LWLIGHT_POINT	 1
#define LWLIGHT_SPOT	 2

#define LWLSHAD_OFF	 0
#define LWLSHAD_RAYTRACE 1
#define LWLSHAD_MAP	 2



/*
 * -----------------
 * Camera Information -- "LW Camera Info"
 *
 * Given the item ID for a camera, a client can get some information
 * specific to the camera.
 */
typedef struct st_LWCameraInfo {
	double		(*zoomFactor) (LWItemID, LWTime);
	double		(*focalLength) (LWItemID, LWTime);
	double		(*focalDistance) (LWItemID, LWTime);
	double		(*fStop) (LWItemID, LWTime);
	double		(*blurLength) (LWItemID, LWTime);
	void		(*fovAngles) (LWItemID, LWTime,
				      double *horizontal, double *vertical);
} LWCameraInfo;



/*
 * -----------------
 * Scene Information -- "LW Scene Info"
 *
 * This structure contains information about the scene itself which is
 * all strictly read-only.
 */
typedef struct st_LWSceneInfo {
	const char	*name;
	const char	*filename;
	int		 numPoints;
	int		 numPolygons;
	int		 renderType;
	int		 renderOpts;
	LWFrame		 frameStart;
	LWFrame		 frameEnd;
	LWFrame		 frameStep;
	double		 framesPerSecond;
	int		 frameWidth;
	int		 frameHeight;
	double		 pixelAspect;
	int		 minSamplesPerPixel;
	int		 maxSamplesPerPixel;
	int		 limitedRegion[4];	/* x0, y0, x1, y1 */
} LWSceneInfo;

/*
 * `renderType' can be one of the following values.
 */
#define LWRTYPE_WIRE		0
#define LWRTYPE_QUICK		1
#define LWRTYPE_REALISTIC	2

/*
 * `renderOpts' is a combination of bits for different rendering options.
 * EVENFIELDS is set only if field rendering is on and the first line of
 * the output image is from the field that comes first in time.
 */
#define LWROPT_SHADOWTRACE	(1<<0)
#define LWROPT_REFLECTTRACE	(1<<1)
#define LWROPT_REFRACTTRACE	(1<<2)
#define LWROPT_FIELDS		(1<<3)
#define LWROPT_EVENFIELDS	(1<<4)
#define LWROPT_MOTIONBLUR	(1<<5)
#define LWROPT_DEPTHOFFIELD	(1<<6)
#define LWROPT_LIMITEDREGION	(1<<7)



/*
 * -----------------
 * Image List Information -- "LW Image List"
 *
 * These functions allow the plugin client to traverse LightWave's image
 * list and use these images for their own purposes.
 *
 * first	- gets the first image in the list, null if none.
 * next		- gets the next image after the given one, null if none.
 * load		- loads a file as an image and adds it to the list.
 * name		- gets the user's name for the image.
 * filename	- gets the filename for the loaded image.  This is the
 *		  value that should be stored for later retrieval of the
 *		  image using `load.'
 * isColor	- returns true if the image has color data or false if
 *		  only greyscale.
 * needAA	- informs LightWave that the image requires antialiasing.
 *		  This can only be called from a shader's `init' function.
 * size		- gets the width and height of the image in pixels.
 * luma		- gets the greyscale value of the image from 0-255.  If
 *		  this is a color source image the value returned is the
 *		  NTSC luminence.
 * RGB		- gets the RGB color of the image from 0-255 at the given
 *		  pixel.
 * lumaSpot, RGBSpot
 *		- gets the floating point greyscale or color value of the
 *		  image for a spot of the given diameter at the given
 *		  center.  These functions can only be called during pixel
 *		  shading and `needAA' must have been called during
 *		  rendering initialization.  If the spot size is small and
 *		  `blend' is true, the color value will be interpolated
 *		  from between image pixels.
 */
typedef void *		 LWImageID;

typedef struct st_LWImageList {
	LWImageID	(*first) (void);
	LWImageID	(*next) (LWImageID);
	LWImageID	(*load) (const char *);
	const char *	(*name) (LWImageID);
	const char *	(*filename) (LWImageID);
	int		(*isColor) (LWImageID);
	void		(*needAA) (LWImageID);
	void		(*size) (LWImageID, int *w, int *h);
	BufferValue	(*luma) (LWImageID, int x, int y);
	void		(*RGB)  (LWImageID, int x, int y, BufferValue[3]);
	double		(*lumaSpot) (LWImageID, double x, double y,
				     double spotSize, int blend);
	void		(*RGBSpot)  (LWImageID, double x, double y,
				     double spotSize, int blend, double[3]);
} LWImageList;



/*
 * -----------------
 * Envelope -- "LW Envelope Hander"
 *
 * Envelopes can be used by plugin clients to manage their own single-channel
 * time values.  Envelopes are an instance which can be created/destroyed,
 * loaded and saved just like any other, but the handler is provided globally
 * from within LightWave.
 */
typedef struct st_LWEnvelopeHandler {
	LWInstHandler	  inst;
	double		(*evaluate) (LWInstance, LWTime);
	void		(*interface) (LWInstance);
} LWEnvelopeHandler;



/*
 * -----------------
 * Global memory pool for rendering -- "Global Render Memory"
 *
 * The Global Pool provides access to shared memory that can be used while
 * rendering.  This has two main uses:  The first is for read-only tables,
 * like trig or random noise lookup tables which can be shared by textures.
 * The second is for communication areas for textures that wish to cooperate
 * in terms of sharing computed values on a per-pixel basis.  LightWave does
 * nothing to manage this shared pool expect to clear it out after rendering.
 * The memory chunks are identified by null-terminated character strings
 * which can be anything.
 */
typedef void *		MemChunk;

typedef struct st_GlobalPool {
	MemChunk	(*first) (void);
	MemChunk	(*next) (MemChunk);
	const char *	(*ID)   (MemChunk);
	int		(*size) (MemChunk);
	MemChunk	(*find)   (const char *ID);
	MemChunk	(*create) (const char *ID, int size);
} GlobalPool;


#endif
