/*
 * LWMOD.H -- LightWave Modeling plugin server interfaces.
 *
 * written by Stuart Ferguson
 * last revision  12/6/94
 */
#ifndef LW_MOD_H
#define LW_MOD_H

#include <moni.h>
#include <lwdyna.h>
#include <lwbase.h>


	/*
	 * ================================
	 * COMMON DEFINITIONS
	 *
	 * These types and definitions are common to the various plugins
	 * and global accessors.
	 */


/*
 * EltOpSelect is a selection mode for operations.  GLOBAL indicates that
 * all elements, both selected and unselected, will be part of the operation.
 * USER means that only those elements selected (explicitly or implicitly)
 * by the user will be included.  DIRECT mean that directly selected 
 * elements will be affected.
 */
typedef int		 EltOpSelect;
#define OPSEL_GLOBAL	 0
#define OPSEL_USER	 1
#define OPSEL_DIRECT	 2

/*
 * EltOpLayer codes are used to select which layers will be affected by an
 * operation.  In addition to the defined values below, codes from 101 to
 * 110 can be used to select the individual layers 1 through 10.
 */
typedef int		 EltOpLayer;
#define OPLYR_PRIMARY	 0	/* one active layer affected by data edit */
#define OPLYR_FG	 1	/* active layers */
#define OPLYR_BG	 2	/* inactive background layers */
#define OPLYR_SELECT	 3	/* displayed layers, fg and bg */
#define OPLYR_ALL	 4	/* all layers */
#define OPLYR_EMPTY	 5	/* layers devoid of data */
#define OPLYR_NONEMPTY	 6	/* layers containing data */




	/*
	 * ================================
	 * SERVER CLASSES
	 *
	 * The server classes are given by the class name(s) and describe
	 * the data types and structures used in its operation.
	 */


/*
 * -----------------
 * 3D Mesh Data Editing -- "MeshDataEdit"
 *
 * The Mesh Edit interface provides a somewhat limited form of editing 
 * existing layer data though point and polygon enumeration functions.  The 
 * client can start an edit requesting a user data area of a certain size
 * for all points and polygons.  It can then traverse the point and polygon
 * lists as often as it wants, calling the edit functions inside or outside
 * the enumeration callbacks.
 */
typedef struct st_Vertex	*PntID;
typedef struct st_Polygon	*PolID;
typedef struct st_EditState	*EditStateRef;

/*
 * The PointInfo and PolygonInfo structures describe points and polygons in
 * the source data.  These are the data elements already in the layers when
 * the edit operation is started.  Every point and polygon has an ID, a
 * userData pointer, a layer number and flags.  The ID uniquely identifies
 * each element and is the reference used for manipulating them.  The
 * userData is a memory block which has been allocated for the client
 * specifically for this element.  This is an area where the client can store
 * computed values for points and polygon while it operates on them.  The
 * size of the point and polygon user data areas is set at the start of
 * editing.  The layer number is just the number of the layer where the
 * element is located (0-9 currently).  All elements have flags bits for
 * selection and deletion.  The SELECT bit is set if the element matched the
 * selection criterion from the start of the edit, and the delete bit is set
 * if the element has been deleted in this session.  Except for userData, the
 * contents of info structures cannot be modified.
 *
 * In addition to the common parts, a PointInfo struct also includes the
 * point position as XYZ coordinates.
 */
typedef struct st_PointInfo {
	PntID		 pnt;
	void		*userData;
	int		 layer;
	int		 flags;
	double		 position[3];
} PointInfo;

/*
 * The PolygonInfo struct contains the common information for all elements
 * and some additional flags for different polygon types.  The polygon shape
 * is given as a number of points and an array of their IDs, plus a surface
 * name.
 */
typedef struct st_PolygonInfo {
	PolID		 pol;
	void		*userData;
	int		 layer;
	int		 flags;
	int		 numPnts;
	const PntID	*points;
	const char	*surface;
} PolygonInfo;

/*
 * Flag bits for points and polygons.  The SELECT and DELETE state
 * flags will be set for both points and polygons, but the remaining flags
 * relate only to polygons.  CCEND and CCSTART are set if the polygon has
 * continuity points at either end.  CURVE is set if this is a curve (it
 * is a face if this is clear).  DETAIL is set if the polygon is a detail.
 */
#define PPDF_SELECT	(1<<0)
#define PPDF_DELETE	(1<<1)
#define PPDF_CCEND	(1<<2)
#define PPDF_CCSTART	(1<<3)
#define PPDF_CURVE	(1<<4)
#define PPDF_DETAIL	(1<<5)

/*
 * Errors are integer codes returned from functions.  The exceptions are
 * functions which create things and return pointers to them in which
 * case an error is signaled by a null return value.  The BADLAYER
 * error will be returned for an attempt to operate on data not in the
 * primary edit layer.
 */
typedef int		EDError;
#define EDERR_NONE	0
#define EDERR_NOMEMORY	1
#define EDERR_BADLAYER	2
#define EDERR_USERABORT	3

/*
 * These are the prototypes for client-provided enumeration functions for
 * scanning points and polygons in the layers.  The client can select some
 * subset of layers and the scan function will be called for each element
 * with a client pointer and info for the element.  If the client returns
 * an error code (or any non-zero value for that matter) from this
 * function, the scan will be aborted and the error will be returned by
 * the scan entry function.
 */
typedef EDError		PointScanFunc (void *, const PointInfo *);
typedef EDError		PolyScanFunc (void *, const PolygonInfo *);

/*
 * The MeshEditOp structure is used to perform an edit operation.  The
 * internal state of the edit is maintained in the private `state' field
 * which is an argument to each function.  The `done' function must be
 * called to complete or abort the operation.  The other functions
 * may be called in any order to perform the desired edit.  Points and
 * polygons may only be edited which belong to the primary active layer,
 * given by the layer number `layerNum.'  All new data will be added to
 * this layer and changes attempted on data in other layers will fail.
 *
 * All changes for a given edit operation must be made through the
 * appropriate callbacks.  The PointInfo and PolygonInfo structures cannot
 * be modified directly and any attempts to do so will either be futile
 * or catastrophic.  As changes are made they are buffered through the
 * undo mechanism, so they are not reflected in the data until the operation
 * is complete.  For example, if a MeshDataEdit client reads the coordinates
 * of a point and changes them (correctly using the `pntMove' function) and
 * reads the coordinates again, they will be the same as the first time.
 * The coordinates will not change until the edits are sucessfully applied
 * using the `done' function.  "done(state,EDERR_NONE,selm)" will complete
 * and apply the cumulative edits, and done with an error code will abort and
 * discard any changes made.  The `selm' argument is bit flags for how to
 * alter the selection based on editing changes.
 */
typedef struct st_MeshEditOp {
	EditStateRef	  state;
	int		  layerNum;
	void		(*done) (EditStateRef, EDError, int selm);

	/*
	 * Return the number of each type of element in the given
	 * mode.
	 */
	int		(*pointCount) (EditStateRef, EltOpLayer, int mode);
	int		(*polyCount)  (EditStateRef, EltOpLayer, int mode);

	/*
	 * Scan points and polygons in layer data.  The client provides
	 * a scan callback and data pointer and all the requested data
	 * will be scanned.
	 */
	EDError		(*pointScan) (EditStateRef, PointScanFunc *,
				      void *, EltOpLayer);
	EDError		(*polyScan)  (EditStateRef, PolyScanFunc *,
				      void *, EltOpLayer);

	/*
	 * Get info for points and polygons.  Info blocks can be requested
	 * outside of scan callbacks.  The returned pointer is only valid
	 * until the next call to an info function.  The normal vector for
	 * a polygon may also be found.  The `polyNormal' function returns
	 * zero if the polygon has fewer than 3 vertices, or the normal is
	 * degenerate for some reason.  If it returns 1, then the normal
	 * has been written to the caller's vector.
	 */
	PointInfo *	(*pointInfo)  (EditStateRef, PntID);
	PolygonInfo *	(*polyInfo)   (EditStateRef, PolID);
	int		(*polyNormal) (EditStateRef, PolID, double[3]);

	/*
	 * Create new data elements.  Points are created by passing a
	 * vector of X,Y and Z coordinates.  Polygons are created from a
	 * surface name (or null for default), number of points and point
	 * list.  Quadrangles and triangles may also be created using the
	 * default surface and will obey the new data options for two-sided
	 * and triangles only.  The points used in new polygons may be ones
	 * which already existed or new ones created in this edit session.
	 */
	PntID		(*addPoint) (EditStateRef, double *);
	PolID		(*addPoly)  (EditStateRef, const char *, int, const PntID *);
	EDError		(*addQuad)  (EditStateRef, PntID, PntID, PntID, PntID);
	EDError		(*addTri)   (EditStateRef, PntID, PntID, PntID);

	/*
	 * The remaining functions are used to alter existing data.  If
	 * called with elements created in this edit session they will
	 * return BADLAYER.
	 */

	/*
	 * Remove existing data elements.  These will remove points and polygons
	 * from the current data set.
	 */
	EDError		(*remPoint) (EditStateRef, PntID);
	EDError		(*remPoly)  (EditStateRef, PolID);

	/*
	 * Move a point.  The point will be moved to the new coordinates.
	 */
	EDError		(*pntMove) (EditStateRef, PntID, const double *);

	/*
	 * Change polygon surface.  The polygon will be altered to use
	 * the new named surface.
	 */
	EDError		(*polSurf) (EditStateRef, PolID, const char *);

	/*
 	 * Change point list.  The polygon will be changed to have a new
	 * set of points given by the list of IDs.  The PntIDs may be for
	 * existing points or points created this session, but should not
	 * refer to points that will be deleted.
	 */
	EDError		(*polPnts) (EditStateRef, PolID, int, const PntID *);

	/*
	 * Change polygon attributes.  The first mask is the set of attributes
	 * to change and the second is their new values.  Only PPDF_CCEND and
	 * PPDF_CCSTART can currently be modified.
	 */
	EDError		(*polFlag) (EditStateRef, PolID, int, int);
} MeshEditOp;

/*
 * The `selm' argument to the `done' callback provides hints on how to
 * alter the user's selection based on data changes.  A value of zero
 * leaves all directly selected elements selected after the edit.  The
 * CLEARCURRENT flag set will clear the current selected elements, and
 * the SELECTNEW flag will cause any newly created elements to become
 * selected.
 */
#define EDSELM_CLEARCURRENT	(1<<0)
#define EDSELM_SELECTNEW	(1<<1)

/*
 * The `mode' argument to the count functions can have one of the
 * following values.  The function will return the count of all elements,
 * only selected elements or only deleted elements.
 */
#define EDCOUNT_ALL		 0
#define EDCOUNT_SELECT		 1
#define EDCOUNT_DELETE		 2

/*
 * A MeshEditBegin function is provided to initiate a data edit operation.
 * It will create an editing context and return the state and functions for
 * this operation.  The two values are the client data sizes in bytes for
 * points and polygons, respectively.  These may be zero.  The selection
 * option determines what elements are initially selected.
 *
 * A MeshEditBegin pointer is the local data passed to a "MeshDataEdit"
 * class server.  The function is called to start the edit operation and can
 * be called only once for each server activation.
 */
typedef MeshEditOp *	MeshEditBegin (int, int, EltOpSelect);



/*
 * -----------------
 * Command Sequence -- "CommandSequence"
 *
 * Command class servers can process a sequence of commands.  They get the
 * ModCommand struct as their local data and can call the `lookup' function
 * to get the codes for command names.  Armed with codes, they can call
 * the `execute' function any number of times with the command code, the
 * number of arguments and the argument values encoded as DynaValues. 
 * Elements to be altered can be selected using the OPSEL mode.
 *
 * A layer mesh edit operation can be started from a command server, but
 * it should be completed before more commands are performed.
 */
typedef int		CommandCode;

typedef struct st_ModCommand {
	void		 *data;
	const char	 *argument;
	CommandCode	(*lookup)  (void *, const char *cmdName);
	int		(*execute) (void *, CommandCode cmd, int argc,
				    const DynaValue *argv, EltOpSelect);
	MeshEditBegin	 *editBegin;
} ModCommand;




	/*
	 * ================================
	 * GLOBAL DATA
	 *
	 * This section contains descriptions of the global data pointers
	 * which can be accessed from Modeler's global function.  The ID
	 * string for each global is given in quotes.
	 */


/*
 * -----------------
 * Dynamic type conversion function -- "LWM: Dynamic Conversion"
 *
 * The conversion function can be used to translate a dynamic type
 * element to another type.  An error may be returned if the conversion
 * cannot be performed, and hints may be provided when converting
 * strings to integer bitfield or choice values.
 */
typedef int		DynaConvertFunc (const DynaValue *, DynaValue *,
					 const DynaStringHint *);

/*
 * -----------------
 * Dynamic Requester functions -- "LWM: Dynamic Request"
 *
 * The functions may be used to create a dynamic requester, to add
 * controls to it, set and read their values and post the request for
 * the user to modify.
 */
typedef struct st_DynaReqFuncs {
	DynaRequestID	(*create)   (const char *);
	int		(*addCtrl)  (DynaRequestID, const char *,
				     DyReqControlDesc *);
	DynaType	(*ctrlType) (DynaRequestID, int);
	int		(*valueSet) (DynaRequestID, int, DynaValue *);
	int		(*valueGet) (DynaRequestID, int, DynaValue *);
	int		(*post)     (DynaRequestID);
	void		(*destroy)  (DynaRequestID);
} DynaReqFuncs;


/*
 * -----------------
 * Dynamic Monitor progress graph -- "LWM: Dynamic Monitor"
 *
 * Monitors provide feedback on the progress of lengthy operations and
 * allow the user to abort them.  Clients create a monitor instance with a
 * title and then use the monitor as described in the specification for
 * Monitors given in `moni.h'.  When done, the monitor must be released.
 * The create func may return null if the Modeler bar graph is already in
 * use, since there can be only one.
 */
typedef struct st_DynaMonitorFuncs {
	Monitor *	(*create) (const char *, const char *);
	void		(*destroy) (Monitor *);
} DynaMonitorFuncs;


/*
 * -----------------
 * Internal state queries -- "LWM: State Query"
 *
 * Modeler's global state can be queried at any time although it can only
 * be altered at specific times.
 *
 * numLayers	- returns total number of data layers in Modeler.
 * layerMask	- returns bits describing the layers included in each of
 *		  the possible EltOpLayer selections.  If mask includes
 *		  layer 1 then bit 0 is set, if it includes layer 2 bit 1
 *		  is set, and so forth.
 * surface	- returns the name of the default surface.
 * bbox		- returns the number of points in the given layer, and if
 *		  minmax is non-null it is filled with the bounding box
 *		  information for the layer (x0, x1, y0, y1, z0, z1).
 */
typedef struct st_StateQueryFuncs {
	int		(*numLayers) (void);
	unsigned int	(*layerMask) (EltOpLayer);
	const char *	(*surface) (void);
	unsigned int	(*bbox) (EltOpLayer, double *minmax);
} StateQueryFuncs;


/*
 * -----------------
 * Surfaces List access -- "LWM: Surface List"
 *
 * The surface list can be accessed and modified by clients at any time.
 * Clients may add, rename and modify the contents of surfaces, but there
 * is no capability to remove them.  Note that adding surfaces or renaming
 * them will alter the relative order of surfaces in the list.
 *
 *   next	Returns the name of the surface after the given one.  If the
 *		given name is a null pointer, this returns the first surface.
 *		If the given name is the last surface, this returns null.
 *   create	Creates a surface of the given name with no data.
 *   rename	Changes the name of a surface, reordering it in the list.
 *   getData	Returns the size and contents of the surface data parameters
 *		for the named surface.
 *   setData	Sets the data parameter block to the given size and contents.
 */
typedef struct st_SurfaceListFuncs {
	const char *	(*next)    (const char *name);
	void		(*create)  (const char *name);
	void		(*rename)  (const char *name, const char *newName);
	void *		(*getData) (const char *name, int *size);
	void		(*setData) (const char *name, int size, void *data);
} SurfaceListFuncs;


/*
 * -----------------
 * Outline Font List access -- "LWM: Font List"
 *
 * The outline font list can be accessed and modified by clients at
 * any time.  Note that altering the list may affect stored font choices
 * in dynamic requesters.
 *
 *   count	returns the total number of fonts in the list.
 *   index	returns the list index for a named font, -1 if not found.
 *   name	returns the name of a font given its list index, NULL if
 *		out of range.
 *   load	loads the given file as a Type-1 font and returns the new
 *		list index.  -1 is returned for errors.
 *   clear	removes the font at the given index from the list.
 */
typedef struct st_FontListFuncs {
	int		(*count) (void);
	int		(*index) (const char *name);
	const char *	(*name)  (int index);
	int		(*load)  (const char *filename);
	void		(*clear) (int index);
} FontListFuncs;


#endif
