/*
 * LWDYNA.H -- Dynamic data types and dynamic requester for use with
 * Modeler plug-in servers.
 *
 * written by Stuart Ferguson
 * last revision  11/21/94
 */
#ifndef LW_DYNA_H
#define LW_DYNA_H


/*
 * Dynamic types are abstract datatypes for values which can be passed
 * to commands from CommandSequence plug-ins, and for values which may
 * by displayed and edited in a Dynamic Requester.  Each entry lists
 * the value typed used to encode the value for the type.
 */
typedef enum en_DynaType {
	DY_NULL,	/* none		*/
	DY_STRING,	/* DyValString	*/
	DY_INTEGER,	/* DyValInt	*/
	DY_FLOAT,	/* DyValFloat	*/
	DY_DISTANCE,	/* DyValFloat	-- value in meters		*/
	DY_VINT,	/* DyValIVector	*/
	DY_VFLOAT,	/* DyValFVector	*/
	DY_VDIST,	/* DyValFVector	-- values in meters		*/
	DY_BOOLEAN,	/* DyValInt	-- zero or non-zero		*/
	DY_CHOICE,	/* DyValInt	-- zero to N minus one		*/
	DY_SURFACE,	/* DyValString	*/
	DY_FONT,	/* DyValInt	-- font number (0 to N-1)	*/
	DY_TEXT,	/* none		*/
	DY_LAYERS,	/* DyValInt	-- layer bit mask		*/
	DY_CUSTOM,	/* 		-- varies			*/
	DY__LAST
} DynaType;


/*
 * Values are defined for a dynamic type using structure whose first field
 * is a DynaType code for the type of value and followed by different
 * fields which hold the value in the form appropriate to the type.
 * Default values are used by dynamic requesters as a RESET value.
 */
typedef struct st_DyValString {
	DynaType	 type;
	char		*buf;
	int		 bufLen;	/* will only be non-zero if the
					 * buffer is writable */
} DyValString;
typedef struct st_DyValInt {
	DynaType	 type;
	int		 value;
	int		 defVal;
} DyValInt;
typedef struct st_DyValFloat {
	DynaType	 type;
	double		 value;
	double		 defVal;
} DyValFloat;
typedef struct st_DyValIVector {
	DynaType	 type;
	int		 val[3];
	int		 defVal;
} DyValIVector;
typedef struct st_DyValFVector {
	DynaType	 type;
	double		 val[3];
	double		 defVal;
} DyValFVector;

/*
 * A DynaValue type is the union of all possible value types plus the type
 * code itself for easy parsing.
 */
typedef union un_DynaValue {
	DynaType	 type;
	DyValString	 str;
	DyValInt	 intv;
	DyValFloat	 flt;
	DyValIVector	 ivec;
	DyValFVector	 fvec;
} DynaValue;


/*
 * A Dynamic Requester is dynamic in two ways; it is built dynamically and
 * it allows user editing of dynamic values.
 */
typedef struct st_DynaRequest	*DynaRequestID;


/*
 * Dynamic request control descriptors are structures with a DynaType for
 * the type of data being edited, and additional settings for the display
 * of the value.
 */

typedef struct st_DyReqStringDesc {
	DynaType	 type;		/* must be DY_STRING */
	int		 width;		/* width in characters */
} DyReqStringDesc;

typedef struct st_DyReqChoiceDesc {
	DynaType	 type;		/* must be DY_CHOICE */
	const char     **items;		/* null-terminated array of item text */
	int		 vertical;	/* vertical layout */
} DyReqChoiceDesc;

typedef struct st_DyReqTextDesc {
	DynaType	 type;		/* must be DY_TEXT */
	const char     **text;		/* array of text lines */
} DyReqTextDesc;

typedef union un_DyReqControlDesc {
	DynaType	 type;
	DyReqStringDesc	 string;
	DyReqChoiceDesc	 choice;
	DyReqTextDesc	 text;
} DyReqControlDesc;



/*
 * String conversion hints.  Hints are provided when converting strings to
 * values and visa-versa.  The choice hint is list of strings and values --
 * when the string is found, the value is used.  The list is terminated with
 * a null item string.  The bitfield hint is a list of character codes and
 * bit values which will be ORed into the result if the character is
 * present in the string.  The list is terminated with a zero bitval.
 */
typedef struct st_DyChoiceHint {
	const char	*item;
	int		 value;
} DyChoiceHint;
typedef struct st_DyBitfieldHint {
	char		 code;
	int		 bitval;
} DyBitfieldHint;

typedef struct st_DynaStringHint {
	DyChoiceHint	*chc;
	DyBitfieldHint	*bits;
} DynaStringHint;


/*
 * Error codes returned from the dynamic data type functions.
 */
#define DYERR_NONE		  0
#define DYERR_MEMORY		(-1)
#define DYERR_BADTYPE		(-2)
#define DYERR_BADSEQ		(-3)
#define DYERR_BADCTRLID		(-4)
#define DYERR_TOOMANYCTRL	(-5)
#define DYERR_INTERNAL		(-6)


/*
 * These interfaces are for Modeler's internal use only.  Plug-in clients
 * must use the global data interface described in the `lwmod.h' header.
 */
extern int		DynaConvert (const DynaValue *, DynaValue *,
				     const DynaStringHint *);

extern DynaRequestID	DyReqCreate (const char *);
extern int		DyReqAddControl (DynaRequestID, const char *,
					 DyReqControlDesc *);
extern DynaType		DyReqControlType (DynaRequestID, int);
extern int		DyReqValueSet (DynaRequestID, int, DynaValue *);
extern int		DyReqValueGet (DynaRequestID, int, DynaValue *);
extern int		DyRequest (DynaRequestID);
extern void		DyReqDestroy (DynaRequestID);


#endif
