/*
 * LWBASE.H -- Basic server interfaces for all of LightWave.
 *
 * written by Stuart Ferguson
 * last revision  12/5/94
 */
#ifndef LW_BASE_H
#define LW_BASE_H

#include <plug.h>


	/*
	 * ================================
	 * SERVER CLASSES
	 *
	 * These are the common server classes defined for both LightWave
	 * Layout and LightWave Modeler.  Class names are in double quotes.
	 */


/*
 * -----------------
 * File Requesters -- "FileRequester"
 *
 * The server interface for file requesters is defined in the "freq.h"
 * header file.  Plugin file requesters can replace the default behavior
 * of Modeler and Layout when selecting files for loading and saving.
 */


/*
 * -----------------
 * Image Input and Output -- "ImageLoader" & "ImageSaver"
 *
 * The image input and output conversion servers are fully described in
 * the document "image.serv" and the "image.h" header.  Loaders are used
 * by both the modeler and renderer, and savers are only currently used
 * by the renderer.
 */


/*
 * -----------------
 * Object Import -- "ObjectLoader"
 *
 * The ObjectImport structure is passed to a plugin object loader as its
 * local data.  The loader should attempt to parse the input file.  If it
 * cannot open the file it should set the result field to OBJSTAT_BADFILE,
 * and if it cannot recognize the format, it should set the result to
 * OBJSTAT_NOREC and return.  If it can parse the file it should send the
 * mesh and surface data by calling the callbacks in the semi-sequential
 * order given below.  The void pointer `data' should be the first argument
 * to every callback.
 *
 * BEGIN:
 *  Callback `begin' is called to mark the start of new mesh data.  The
 *  second argument is for special information and should normally be NULL.
 *  (It might be possible to call this more than once, although `done'
 *  would have to be called before calling `begin' a second time.)
 *
 * POINTS:
 *  Callback `numPoints' is called with the total number of points.  Then
 *  `points' is called with 1 or more point coordinates until the total
 *  number of points is reached.  Points are numbered from zero in the order
 *  added, and that implicit index is used to create polygons, and all
 *  points must be added before any polygons may be created.
 *
 * SURFACES:
 *  The callback `surfIndex' is called with a surface name to get a surface
 *  ID number for that surface.  This ID number is used to create polygons.
 *  The function may optionally return a boolean flag to indicate if this
 *  is the first time this surface name has been given an ID.
 *
 * POLYGONS:
 *  For each polygon, the `polygon' function is called with a list of point
 *  indices for the polygon, the number of points, mode flags and a surface
 *  index.  If the DETAIL flag bit is set, then this polygon is a detail
 *  of the last top-level polygon.
 *
 * SURFACE DATA:
 *  A block of raw surface parameters may be assigned to a name by calling
 *  the `surfData' call with the surface name and byte block.
 *
 * DONE:
 *  Called when data transfer is complete.
 *
 * If anything fails, the loader should stop sending data, set the result
 * field to OBJSTAT_FAILED and return.  If `failedLen' is non-zero, the
 * loader may also place a human-readable error message into the buffer
 * pointed to by `failedBuf.'
 */
typedef struct st_ObjectImport {
	int		  result;
	const char	 *filename;
	Monitor		 *monitor;
	char		 *failedBuf;
	int		  failedLen;

	void		 *data;
	void		(*begin) (void *, void *);
	void		(*done) (void *);
	void		(*numPoints) (void *, int total);
	void		(*points) (void *, int numPts, const float *xyz);
	int		(*surfIndex) (void *, const char *name, int *firstTime);
	void		(*polygon) (void *, int numPts, int surf, int flags,
				    const unsigned short *);
	void		(*surfData) (void *, const char *name, int size, void *data);
} ObjectImport;

/*
 * Result codes.
 */
#define OBJSTAT_OK	 0
#define OBJSTAT_NOREC	 1
#define OBJSTAT_BADFILE	 2
#define OBJSTAT_ABORTED	 3
#define OBJSTAT_FAILED	 99

/*
 * Polygon flag bits.
 */
#define OBJPOLF_FACE	  0
#define OBJPOLF_CURVE	 (1<<0)
#define OBJPOLF_DETAIL	 (1<<1)
#define OBJPOLF_STARTCC	 (1<<2)
#define OBJPOLF_ENDCC	 (1<<3)



	/*
	 * ================================
	 * GLOBAL DATA
	 *
	 * This section contains descriptions of the data pointers
	 * accessable from the global functions of both Modeler and Layout.
	 * The string for each object is in double quotes.
	 */

/*
 * -----------------
 * File Requester -- "File Request"
 *
 * Clients can get access to a function which allows them to request
 * filenames from users.  The `hail' string is the title of the request
 * and the `name' & `path' buffers should be filled in with the starting
 * base name and path for the request.  These buffers will be modified
 * and the `fullName' buffer filled with the final complete name for the
 * user-selected file.  `bufLen' is the length of all the passed buffers.
 * The function returns 0 if the user elected to cancel, 1 if the hit Ok,
 * and -1 for any errors.
 */
typedef int		FileReqFunc (const char *hail, char *name, char *path,
				     char *fullName, int buflen);


/*
 * User messages -- "Info Messages"
 *
 * The `error' and `warning' functions will display a message to the
 * user with one or two lines of text.  The second string pointer can
 * be NULL for one-line messages.
 */
typedef struct st_MessageFuncs {
	void		(*info)    (const char *, const char *);
	void		(*error)   (const char *, const char *);
	void		(*warning) (const char *, const char *);
} MessageFuncs;


/*
 * Host Display state -- "Host Display Info"
 *
 * This is the HostDisplayInfo struct defined in the `hdisp.h' header file
 * initialized for the host application's main window.
 */


#endif
